classdef HW3Scenario
%Problem Scenario for ICP Homework 
%  
% H = HW3Scenario()                 - Sets up data structure
% 
% [Diff,RE,ErrMax] = H.Test(Fnom,Ns,noise,Po)
%            % Samples data and calls both student program and my program
%
% Samples = H.Sample(H,Fnom,Ns,noise,Po)
%            % Returns random samples from mesh surface with noise added
%            % Fnom = nominal pose of surface mesh
%            % Ns = Number of good samples
%            % Po = Probability of a sample being an outlier
%
%[cp,ct,cd] = H.Match(TransformedSamples, ... vct3Array of samples
%                      prev_cp,            ... vct3 array of corresponding closest points
%                      prev_ct,            ... integer array of triangle indices for prev_cp
%                      prev_ct)            ... real array of distances for prev_cp
%
%   Performs recursive search through the tree and returns
%    cp       = vct3Array of closest points on mesh to transformed samples
%    ct       = integer array of mesh triangle indices corresponding to cp
%    cd       = distances corresponding to closest points
%
% [cp,ct,cd] = H.InitialMatch(Samples)     % initial samples for H.Match   
%     Samples = set of samples 
%     
% [MedianErr, MeanErr, RMSErr, MaxErr] = DistanceStats(Cd)
%   Computes some useful statistics on Cd 
% 
% Copyright (C) Russell H. Taylor 2013
% For use with CIS I only
% Do not redistribute without written permission from Russell Taylor
    
    properties
        Mesh       	% Surface Mesh
        Tree        % Triangle Cov Tree
        ICP         % ICP Tableau (mostly use just for setup)
    end
    
    methods
        function H = HW3Scenario(MeshTreeFile)
            if nargin>0
                load(MeshTreeFile);
                H.Mesh = Mesh;
                H.Tree = Tree;
            else
                H.Mesh = SurfaceMesh.ReadSurFile('HW3TestMesh.sur');
                display('Building Tree');
                H.Tree = TriangleCovTree(H.Mesh);
                H.ICP = RigidICP(H.Mesh,H.Tree);
            end
        end
        
        function Samples = Sample(H,Fnom,Ns,noise,Po)
            % Returns random samples from mesh surface with noise added
            % Fnom = nominal pose of surface mesh
            % Ns = Number of good samples
            % Po = Probability of a sample being an outlier 
            Samples = vct3Array(Ns);
            BB = H.Tree.Top.BB; % bounding box (use for outliers)
            for i=1:Ns
                if (rand()<Po)
                    s = BB.Sample();
                else
                    [s,t] = H.Mesh.Sample(-noise,noise);
                    Samples(i)=s;
                end;
            end
            Samples=Fnom.Inverse()*Samples;
        end
        
        function[Cp,Ct,Cd] = InitialMatch(H,Samples)
            N = Samples.NumEl();
            Ct = ones(1,N);
            Cd = zeros(1,N);
            Cp = vct3Array(N);
            [~,p,q,r] = H.Mesh.Triangle(1);
            for i = 1:N
                [Cp(i),Cd(i)] = TriangleClosestPoint(Samples(i),p,q,r);
            end;
        end
        
        function [NewCp,NewCt,NewCd] = Match(H,TransformedSamples,Ct,Cd)
            N = TransformedSamples.NumEl();
            NewCp = vct3Array(N);
            NewCt = zeros(N,1);
            NewCd = zeros(N,1);
            fprintf('Matching ');
            for i = 1:N
                fprintf('.');
                Cti = Ct(i);
                [~,p,q,r] = H.Mesh.Triangle(Cti);
                [Cpi,Cdi] = TriangleClosestPoint(TransformedSamples(i),p,q,r);
                [NewCp(i),NewCt(i),NewCd(i)] = H.Tree.SearchForClosestPoint(TransformedSamples(i),Cpi,Cti,Cdi);
            end
            [NewCp(i),NewCt(i),NewCd(i)] = H.Tree.SearchForClosestPoint(TransformedSamples(i),Cpi,Cti,Cdi);
            fprintf('\n');
        end
        
        function [MedianErr, MeanErr, RMSErr, MaxErr] = DistanceStats(H,Cd)
            MedianErr = median(Cd);
            MeanErr = mean(Cd);
            RMSErr = sqrt((Cd*Cd')/size(Cd,2));
            MaxErr = norm(Cd,inf);
        end
        
        function [Stats] = Test(H,Fnom,Ns,noise,Po)
            Samples = H.Sample(Fnom,Ns,noise,Po);
            [Ficp,Cp,Cd] = HW3ICP(H,Samples);
            DeltaF  = Fnom.Inverse()*Ficp;
            [Ax,An] = DeltaF.R.AxisAngle();
            [MedianErr, MeanErr, RMSErr, MaxErr]  = H.DistanceStats(Cd);
            Stats = [An, norm(DeltaF.p), MedianErr, MeanErr, RMSErr, MaxErr];
            fprintf('\nResults of ICP\n');
            fprintf('Angle Err (deg)=%f; Pos Err=%f; Residuals: Median=%f Mean=%f RMS=%f Max=%f\n',...
                     An*180/pi, norm(DeltaF.p), MedianErr, MeanErr, RMSErr, MaxErr);
        end
                
        
    end
    
end

